---
name: sql
description: 효율적인 데이터베이스 쿼리 작성과 성능 최적화 가이드
license: MIT
---

# SQL 개발 스킬

## 기본 원칙
- 인덱스 활용
- 정규화 vs 비정규화
- 쿼리 성능 최적화
- 트랜잭션 관리

## 기본 테이블 구조
```sql
-- 사용자 테이블
CREATE TABLE users (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    email VARCHAR(255) UNIQUE NOT NULL,
    password_hash VARCHAR(255) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_email (email),
    INDEX idx_created_at (created_at)
);

-- 게시글 테이블
CREATE TABLE posts (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT NOT NULL,
    title VARCHAR(200) NOT NULL,
    content TEXT,
    status ENUM('draft', 'published', 'archived') DEFAULT 'draft',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_status (user_id, status),
    INDEX idx_created_at (created_at),
    FULLTEXT INDEX ft_title_content (title, content)
);
```

## 최적화된 쿼리 패턴
```sql
-- 페이지네이션 (OFFSET 대신 커서 기반)
SELECT * FROM posts 
WHERE id > :last_id 
ORDER BY id ASC 
LIMIT 20;

-- 조건부 집계
SELECT 
    COUNT(*) as total_posts,
    COUNT(CASE WHEN status = 'published' THEN 1 END) as published_count,
    AVG(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN 1 ELSE 0 END) as recent_ratio
FROM posts 
WHERE user_id = :user_id;
```

## 필수 체크리스트
- [ ] 적절한 인덱스 설정
- [ ] 외래키 제약조건
- [ ] 트랜잭션 범위 최소화
- [ ] 쿼리 실행계획 확인
- [ ] 백업 및 복구 전략
