---
name: python
description: Python 웹 개발과 데이터 처리를 위한 베스트 프랙티스
license: MIT
---

# Python 개발 스킬

## 기본 원칙
- PEP 8 스타일 가이드 준수
- 타입 힌팅 사용
- 가상환경 관리
- 테스트 주도 개발

## 기본 프로젝트 구조
```
project/
├── src/
│   ├── __init__.py
│   ├── main.py
│   └── utils/
├── tests/
├── requirements.txt
├── .env
└── README.md
```

## 기본 코드 패턴
```python
# main.py
from typing import List, Dict, Optional
import logging
from dataclasses import dataclass
from pathlib import Path

# 설정
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

@dataclass
class DataModel:
    id: int
    name: str
    email: Optional[str] = None

class DataProcessor:
    def __init__(self, config: Dict[str, str]):
        self.config = config
    
    def process_data(self, data: List[Dict]) -> List[DataModel]:
        try:
            return [DataModel(**item) for item in data]
        except Exception as e:
            logger.error(f"Data processing error: {e}")
            raise

if __name__ == "__main__":
    processor = DataProcessor({"env": "production"})
```

## 필수 체크리스트
- [ ] 가상환경 설정
- [ ] requirements.txt 관리
- [ ] 타입 힌팅 사용
- [ ] 예외 처리
- [ ] 로깅 구현

## 팀 컨벤션 체크
- [ ] `pyproject.toml`로 Black/isort/flake8/mypy 설정 통합
- [ ] Poetry로 의존성/스크립트 관리
- [ ] 타입 힌트 100% 적용 + mypy strict 모드
- [ ] pytest + pytest-cov로 커버리지 90% 이상 유지
- [ ] pre-commit 훅에 포매터/린터/테스트 연결
