---
name: php
description: 모던 PHP 웹 개발과 Laravel 프레임워크 활용 가이드
license: MIT
---

# PHP 개발 스킬

## 기본 원칙
- PSR 표준 준수
- Composer 패키지 관리
- OOP 패턴 사용
- 보안 고려사항

## 기본 클래스 구조
```php
<?php

declare(strict_types=1);

namespace App\Services;

use App\Models\User;
use App\Exceptions\ValidationException;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;

class UserService
{
    public function __construct(
        private User $userModel
    ) {}

    public function createUser(array $data): User
    {
        try {
            $validatedData = $this->validateUserData($data);
            
            return $this->userModel->create([
                'name' => $validatedData['name'],
                'email' => $validatedData['email'],
                'password' => Hash::make($validatedData['password'])
            ]);
        } catch (\Exception $e) {
            Log::error('User creation failed', ['error' => $e->getMessage()]);
            throw new ValidationException('사용자 생성에 실패했습니다.');
        }
    }

    private function validateUserData(array $data): array
    {
        $required = ['name', 'email', 'password'];
        
        foreach ($required as $field) {
            if (empty($data[$field])) {
                throw new ValidationException("$field 는 필수 항목입니다.");
            }
        }
        
        return $data;
    }
}
```

## 필수 체크리스트
- [ ] 타입 선언 (strict_types=1)
- [ ] 네임스페이스 사용
- [ ] 의존성 주입 패턴
- [ ] 예외 처리
- [ ] 보안 검증 (XSS, SQL Injection)

## 팀 컨벤션 체크
- [ ] Composer 오토로딩 PSR-4 규칙 준수
- [ ] PHPStan level 8 통과 및 CI 게이트 적용
- [ ] Doctrine ORM 또는 Laravel Eloquent 사용 시 스키마 문서화
- [ ] Symfony 컴포넌트/Contracts 적극 활용
- [ ] PHP 8+ 문법(속성, union type, match 등) 우선 적용

## 에러 핸들링 패턴

### 1. Custom Exception with Context
```php
<?php

declare(strict_types=1);

namespace App\Exceptions;

class BusinessException extends \Exception
{
    private array $context;
    
    public function __construct(
        string $message,
        array $context = [],
        int $code = 0,
        ?\Throwable $previous = null
    ) {
        parent::__construct($message, $code, $previous);
        $this->context = $context;
    }
    
    public function getContext(): array
    {
        return $this->context;
    }
    
    public function toArray(): array
    {
        return [
            'message' => $this->getMessage(),
            'code' => $this->getCode(),
            'context' => $this->context,
            'file' => $this->getFile(),
            'line' => $this->getLine()
        ];
    }
}
```

### 2. Result Pattern with Either Type
```php
<?php

declare(strict_types=1);

namespace App\Utils;

class Result
{
    private bool $success;
    private mixed $data;
    private ?string $error;
    
    private function __construct(bool $success, mixed $data, ?string $error)
    {
        $this->success = $success;
        $this->data = $data;
        $this->error = $error;
    }
    
    public static function success(mixed $data): self
    {
        return new self(true, $data, null);
    }
    
    public static function failure(string $error): self
    {
        return new self(false, null, $error);
    }
    
    public function isSuccess(): bool
    {
        return $this->success;
    }
    
    public function getData(): mixed
    {
        return $this->data;
    }
    
    public function getError(): ?string
    {
        return $this->error;
    }
    
    public function map(callable $fn): self
    {
        if (!$this->success) {
            return $this;
        }
        
        try {
            return self::success($fn($this->data));
        } catch (\Exception $e) {
            return self::failure($e->getMessage());
        }
    }
}

// 사용 예시
function createUser(array $data): Result
{
    try {
        $validatedData = validateUserData($data);
        $user = User::create($validatedData);
        
        return Result::success($user);
    } catch (ValidationException $e) {
        return Result::failure("유효성 검사 실패: " . $e->getMessage());
    } catch (\Exception $e) {
        Log::error('User creation failed', ['error' => $e->getMessage()]);
        return Result::failure("사용자 생성 중 오류가 발생했습니다.");
    }
}
```

## 테스팅 예제

### PHPUnit Test
```php
<?php

declare(strict_types=1);

namespace Tests\Unit\Services;

use App\Services\UserService;
use App\Models\User;
use App\Exceptions\ValidationException;
use PHPUnit\Framework\TestCase;
use Illuminate\Foundation\Testing\RefreshDatabase;

class UserServiceTest extends TestCase
{
    use RefreshDatabase;
    
    private UserService $userService;
    
    protected function setUp(): void
    {
        parent::setUp();
        $this->userService = new UserService(new User());
    }
    
    public function test_create_user_successfully(): void
    {
        $userData = [
            'name' => 'Test User',
            'email' => 'test@example.com',
            'password' => 'password123'
        ];
        
        $result = $this->userService->createUser($userData);
        
        $this->assertTrue($result->isSuccess());
        $this->assertInstanceOf(User::class, $result->getData());
        $this->assertDatabaseHas('users', ['email' => 'test@example.com']);
    }
    
    public function test_create_user_with_invalid_data(): void
    {
        $userData = [
            'name' => '',
            'email' => 'invalid-email'
        ];
        
        $result = $this->userService->createUser($userData);
        
        $this->assertFalse($result->isSuccess());
        $this->assertStringContains('유효성 검사 실패', $result->getError());
    }
    
    /**
     * @dataProvider invalidUserDataProvider
     */
    public function test_validation_with_data_provider(array $data, string $expectedError): void
    {
        $result = $this->userService->createUser($data);
        
        $this->assertFalse($result->isSuccess());
        $this->assertStringContains($expectedError, $result->getError());
    }
    
    public function invalidUserDataProvider(): array
    {
        return [
            'empty name' => [['name' => '', 'email' => 'test@example.com'], 'name'],
            'invalid email' => [['name' => 'Test', 'email' => 'invalid'], 'email'],
            'missing password' => [['name' => 'Test', 'email' => 'test@example.com'], 'password'],
        ];
    }
}
```

## 프로젝트 구조 예시

```
my-php-project/
├── app/
│   ├── Console/            # Artisan 명령어
│   ├── Exceptions/         # 커스텀 예외
│   │   ├── BusinessException.php
│   │   └── ValidationException.php
│   ├── Http/
│   │   ├── Controllers/    # 컨트롤러
│   │   ├── Middleware/     # 미들웨어
│   │   ├── Requests/       # Form Request 클래스
│   │   └── Resources/      # API 리소스
│   ├── Models/             # Eloquent 모델
│   ├── Services/           # 비즈니스 로직
│   │   ├── UserService.php
│   │   └── PaymentService.php
│   ├── Utils/              # 유틸리티 클래스
│   │   ├── Result.php
│   │   └── Validator.php
│   └── Providers/          # 서비스 프로바이더
├── tests/
│   ├── Feature/            # 기능 테스트
│   ├── Unit/              # 단위 테스트
│   └── TestCase.php       # 기본 테스트 클래스
├── config/                 # 설정 파일
├── database/
│   ├── migrations/         # 데이터베이스 마이그레이션
│   ├── seeders/           # 데이터 시드
│   └── factories/         # 모델 팩토리
├── resources/
│   ├── views/             # Blade 템플릿
│   └── lang/              # 다국어 파일
├── routes/
│   ├── web.php            # 웹 라우트
│   └── api.php            # API 라우트
├── composer.json          # Composer 의존성
├── phpunit.xml           # PHPUnit 설정
├── .env.example          # 환경변수 예시
└── README.md
```
