---
name: javascript
description: 모던 JavaScript ES6+ 문법과 웹 애플리케이션 개발 패턴
license: MIT
---

# JavaScript 개발 스킬

## 기본 원칙
- ES6+ 문법 사용
- 함수형 프로그래밍 패턴
- 비동기 처리 (async/await)
- 모듈화 구조

## 기본 구조
```javascript
// 모듈 임포트
import { API_BASE_URL } from './config.js';

// 상수 정의
const CONSTANTS = {
    API_TIMEOUT: 5000,
    RETRY_COUNT: 3
};

// 클래스 정의
class DataManager {
    constructor(apiUrl) {
        this.apiUrl = apiUrl;
    }

    async fetchData(endpoint) {
        try {
            const response = await fetch(`${this.apiUrl}/${endpoint}`);
            return await response.json();
        } catch (error) {
            console.error('Data fetch error:', error);
            throw error;
        }
    }
}

// 함수 정의
const debounce = (fn, delay) => {
    let timeoutId;
    return (...args) => {
        clearTimeout(timeoutId);
        timeoutId = setTimeout(() => fn.apply(this, args), delay);
    };
};

export { DataManager, debounce };

## 에러 핸들링 패턴

### 1. Try-Catch with Custom Errors
```javascript
class APIError extends Error {
    constructor(message, status, endpoint) {
        super(message);
        this.name = 'APIError';
        this.status = status;
        this.endpoint = endpoint;
    }
}

async function fetchWithRetry(url, options = {}, maxRetries = 3) {
    for (let attempt = 1; attempt <= maxRetries; attempt++) {
        try {
            const response = await fetch(url, options);
            
            if (!response.ok) {
                throw new APIError(
                    `HTTP ${response.status}: ${response.statusText}`,
                    response.status,
                    url
                );
            }
            
            return await response.json();
        } catch (error) {
            console.warn(`Attempt ${attempt} failed:`, error.message);
            
            if (attempt === maxRetries) {
                throw error;
            }
            
            await new Promise(resolve => setTimeout(resolve, 1000 * attempt));
        }
    }
}
```

### 2. Promise Error Boundary
```javascript
const safeAsync = (asyncFn) => {
    return async (...args) => {
        try {
            return { data: await asyncFn(...args), error: null };
        } catch (error) {
            console.error('Async operation failed:', error);
            return { data: null, error: error.message };
        }
    };
};

// 사용 예시
const safeUserFetch = safeAsync(fetchUserData);
const { data: user, error } = await safeUserFetch(userId);

if (error) {
    showErrorMessage(error);
} else {
    updateUI(user);
}
```

## 테스팅 예제

### Jest Unit Test
```javascript
// user-service.test.js
import { UserService } from './user-service.js';

describe('UserService', () => {
    let userService;
    
    beforeEach(() => {
        userService = new UserService('https://api.example.com');
    });
    
    test('should fetch user successfully', async () => {
        // Mock fetch
        global.fetch = jest.fn(() =>
            Promise.resolve({
                ok: true,
                json: () => Promise.resolve({ id: 1, name: 'Test User' })
            })
        );
        
        const user = await userService.getUser(1);
        
        expect(user).toEqual({ id: 1, name: 'Test User' });
        expect(fetch).toHaveBeenCalledWith('https://api.example.com/users/1');
    });
    
    test('should handle API errors', async () => {
        global.fetch = jest.fn(() =>
            Promise.resolve({
                ok: false,
                status: 404,
                statusText: 'Not Found'
            })
        );
        
        await expect(userService.getUser(999))
            .rejects
            .toThrow('HTTP 404: Not Found');
    });
});
```

## 프로젝트 구조 예시

```
my-js-project/
├── src/
│   ├── components/          # UI 컴포넌트
│   │   ├── Header/
│   │   │   ├── Header.js
│   │   │   ├── Header.test.js
│   │   │   └── Header.css
│   │   └── UserCard/
│   ├── services/           # API 서비스
│   │   ├── user-service.js
│   │   └── api-client.js
│   ├── utils/              # 유틸리티 함수
│   │   ├── debounce.js
│   │   ├── validation.js
│   │   └── formatters.js
│   ├── config/             # 설정 파일
│   │   ├── constants.js
│   │   └── environment.js
│   └── app.js              # 메인 앱
├── tests/
│   ├── __mocks__/          # Mock 파일
│   ├── fixtures/           # 테스트 데이터
│   └── setup.js            # 테스트 설정
├── public/                 # 정적 파일
├── .eslintrc.js           # ESLint 설정
├── jest.config.js         # Jest 설정
├── package.json
└── README.md
```
```

## 필수 체크리스트
- [ ] Strict mode 사용
- [ ] 에러 핸들링
- [ ] 비동기 처리
- [ ] 메모리 누수 방지
- [ ] 성능 최적화

## 팀 컨벤션 체크
- [ ] TypeScript와 연동 시 `tsconfig.json` strict 옵션 강제
- [ ] ESLint(Airbnb) + Prettier 통합 설정 유지
- [ ] Husky pre-commit 훅에서 lint/test 실행
- [ ] webpack-bundle-analyzer 등으로 번들 크기 추적
- [ ] CI에서 Jest + Testing Library 커버리지 리포트 생성
