---
name: html
description: HTML 구조화 마크업과 시맨틱 웹 개발을 위한 베스트 프랙티스와 코드 패턴
license: MIT
---

# HTML 개발 스킬

## 기본 원칙
- 시맨틱 HTML5 요소 사용
- 접근성(Accessibility) 고려
- SEO 친화적 구조
- 크로스 브라우저 호환성

## 기본 문서 구조
```html
<!DOCTYPE html>
<html lang="ko">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>페이지 제목</title>
    <meta name="description" content="페이지 설명">
</head>
<body>
    <header>
        <nav></nav>
    </header>
    <main>
        <section></section>
    </main>
    <footer></footer>
</body>
</html>
```

## 필수 체크리스트
- [ ] DOCTYPE 선언
- [ ] lang 속성 설정
- [ ] 메타태그 완성
- [ ] 시맨틱 태그 사용
- [ ] alt 속성 추가
- [ ] 적절한 heading 구조

## 접근성 패턴

### 1. ARIA 레이블과 역할
```html
<!-- 네비게이션 메뉴 -->
<nav role="navigation" aria-label="메인 메뉴">
    <ul>
        <li><a href="/" aria-current="page">홈</a></li>
        <li><a href="/about">소개</a></li>
    </ul>
</nav>

<!-- 폼 with 에러 처리 -->
<form novalidate>
    <div class="form-group">
        <label for="email">이메일 *</label>
        <input type="email" id="email" name="email" 
               required aria-describedby="email-error" 
               aria-invalid="false">
        <div id="email-error" class="error-message" aria-live="polite"></div>
    </div>
    
    <button type="submit" aria-describedby="submit-help">
        계정 생성
    </button>
    <div id="submit-help" class="help-text">
        이메일로 확인 링크를 보내드립니다.
    </div>
</form>
```

### 2. 반응형 이미지
```html
<!-- 반응형 이미지 with fallback -->
<picture>
    <source media="(min-width: 800px)" 
            srcset="large.webp 1x, large-2x.webp 2x" 
            type="image/webp">
    <source media="(min-width: 400px)" 
            srcset="medium.webp 1x, medium-2x.webp 2x" 
            type="image/webp">
    <img src="small.jpg" 
         srcset="small-2x.jpg 2x"
         alt="제품 사진: 빨간색 운동화"
         loading="lazy"
         width="300" height="200">
</picture>
```

## 성능 최적화

### Critical CSS 인라인
```html
<head>
    <!-- Critical CSS 인라인 -->
    <style>
        /* Above-the-fold 스타일만 인라인 */
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif; }
        .header { background: #fff; position: fixed; top: 0; width: 100%; }
        .hero { min-height: 60vh; display: flex; align-items: center; }
    </style>
    
    <!-- 나머지 CSS는 비동기 로드 -->
    <link rel="preload" href="styles.css" as="style" onload="this.onload=null;this.rel='stylesheet'">
    <noscript><link rel="stylesheet" href="styles.css"></noscript>
</head>
```

### 리소스 최적화
```html
<!-- DNS prefetch -->
<link rel="dns-prefetch" href="//fonts.googleapis.com">
<link rel="dns-prefetch" href="//api.example.com">

<!-- 중요 리소스 preload -->
<link rel="preload" href="font.woff2" as="font" type="font/woff2" crossorigin>
<link rel="preload" href="hero-image.jpg" as="image">

<!-- 다음 페이지 prefetch -->
<link rel="prefetch" href="/about">
```

## 프로젝트 구조 예시

```
my-web-project/
├── public/
│   ├── index.html          # 메인 페이지
│   ├── about.html          # 소개 페이지
│   ├── contact.html        # 연락처 페이지
│   └── assets/
│       ├── css/
│       │   ├── critical.css    # Critical path CSS
│       │   ├── main.css        # 메인 스타일
│       │   └── responsive.css  # 반응형 스타일
│       ├── js/
│       │   ├── main.js         # 메인 스크립트
│       │   ├── utils.js        # 유틸리티
│       │   └── components/     # 컴포넌트별 스크립트
│       ├── images/
│       │   ├── optimized/      # 최적화된 이미지
│       │   └── icons/          # SVG 아이콘
│       └── fonts/              # 웹폰트
├── src/                        # 소스 파일 (빌드 전)
│   ├── templates/              # HTML 템플릿
│   ├── scss/                   # SCSS 파일
│   └── js/                     # 원본 JS 파일
├── tools/
│   ├── build.js               # 빌드 스크립트
│   ├── optimize-images.js     # 이미지 최적화
│   └── accessibility-test.js  # 접근성 테스트
├── .htmlhintrc                # HTML 린트 설정
├── package.json
└── README.md
```
