---
name: docker
description: Docker 컨테이너화와 멀티 스테이지 빌드 최적화 가이드
license: MIT
---

# Docker 개발 스킬

## 기본 원칙
- 멀티 스테이지 빌드
- 레이어 최적화
- 보안 고려사항
- 이미지 크기 최소화

## Dockerfile 템플릿
```dockerfile
# Node.js 앱용 멀티스테이지 Dockerfile
FROM node:18-alpine AS builder

WORKDIR /app
COPY package*.json ./
RUN npm ci --only=production && npm cache clean --force

FROM node:18-alpine AS runtime

# 보안: 비루트 사용자 생성
RUN addgroup -g 1001 -S nodejs && \
    adduser -S nextjs -u 1001

WORKDIR /app

# 필요한 파일만 복사
COPY --from=builder /app/node_modules ./node_modules
COPY --chown=nextjs:nodejs . .

USER nextjs

EXPOSE 3000

HEALTHCHECK --interval=30s --timeout=3s --start-period=5s --retries=3 \
    CMD node healthcheck.js

CMD ["node", "server.js"]
```

## docker-compose.yml 템플릿
```yaml
version: '3.8'

services:
  app:
    build: 
      context: .
      dockerfile: Dockerfile
    ports:
      - "3000:3000"
    environment:
      - NODE_ENV=production
      - DATABASE_URL=${DATABASE_URL}
    depends_on:
      - db
    restart: unless-stopped
    healthcheck:
      test: ["CMD", "node", "healthcheck.js"]
      interval: 30s
      timeout: 10s
      retries: 3

  db:
    image: postgres:15-alpine
    environment:
      POSTGRES_DB: ${DB_NAME}
      POSTGRES_USER: ${DB_USER}
      POSTGRES_PASSWORD: ${DB_PASSWORD}
    volumes:
      - postgres_data:/var/lib/postgresql/data
    restart: unless-stopped

volumes:
  postgres_data:
```

## 필수 체크리스트
- [ ] .dockerignore 파일 작성
- [ ] 비루트 사용자 사용
- [ ] 헬스체크 구현
- [ ] 보안 스캔 실행
- [ ] 이미지 크기 최적화
