#!/bin/bash

# 🚀 Claude Skills 동기화 스크립트
# 로컬 스킬 템플릿을 Claude Skills 디렉토리로 동기화합니다.

set -euo pipefail

# ========================================
# 📋 설정 변수
# ========================================

# 기본 설정
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
SOURCE_DIR="${SCRIPT_DIR}/mnt/skills"
DEFAULT_TARGET_DIR="${HOME}/.claude/skills"
LOG_FILE=""
DRY_RUN=false
VERBOSE=false

# 색상 코드
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
PURPLE='\033[0;35m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

# 스킬 목록 (25개 언어/프레임워크)
LANGUAGES=(
    "html" "css" "javascript" "typescript" "react" "vue" "angular"
    "php" "python" "nodejs" "java" "csharp" "go" "rust" "cpp"
    "sql" "docker" "kubernetes" "bash" "powershell"
    "json" "xml" "yaml" "laravel" "django" "flask" "express" "wordpress"
    "general"
)

# ========================================
# 📋 유틸리티 함수
# ========================================

print_header() {
    echo -e "${CYAN}════════════════════════════════════════${NC}"
    echo -e "${CYAN}🚀 Claude Skills 동기화 도구${NC}"
    echo -e "${CYAN}════════════════════════════════════════${NC}"
    echo ""
}

log() {
    local level="$1"
    shift
    local message="$*"
    local timestamp=$(date '+%Y-%m-%d %H:%M:%S')
    
    case "$level" in
        "INFO")
            echo -e "${GREEN}[INFO]${NC} $message"
            ;;
        "WARN")
            echo -e "${YELLOW}[WARN]${NC} $message"
            ;;
        "ERROR")
            echo -e "${RED}[ERROR]${NC} $message"
            ;;
        "DEBUG")
            if [ "$VERBOSE" = true ]; then
                echo -e "${PURPLE}[DEBUG]${NC} $message"
            fi
            ;;
    esac
    
    # 로그 파일이 설정되어 있으면 기록
    if [ -n "$LOG_FILE" ]; then
        echo "[$timestamp][$level] $message" >> "$LOG_FILE"
    fi
}

show_help() {
    cat << EOF
🚀 Claude Skills 동기화 도구

사용법:
  $0 [옵션]

옵션:
  --target <경로>     동기화 대상 경로 (기본값: ~/.claude/skills)
  --dry-run           실제 복사 없이 수행 예정 작업만 출력
  --verbose           상세 로그 출력
  -h, --help          이 도움말 표시

예시:
  $0                                    # 기본 경로로 동기화
  $0 --dry-run                         # 변경사항 미리보기
  $0 --target ~/my-claude-skills       # 커스텀 경로로 동기화
  $0 --verbose                         # 상세 로그와 함께 동기화

환경변수:
  CLAUDE_SKILLS_ROOT   기본 대상 경로 오버라이드

스킬 목록 (${#LANGUAGES[@]}개):
  $(printf "  %s\n" "${LANGUAGES[@]}" | column -c 80)

EOF
}

validate_environment() {
    log "DEBUG" "환경 검증 시작"
    
    # 소스 디렉토리 확인
    if [ ! -d "$SOURCE_DIR" ]; then
        log "ERROR" "소스 디렉토리를 찾을 수 없습니다: $SOURCE_DIR"
        log "INFO" "mnt.zip 파일을 압축 해제하시겠습니까? (y/N)"
        read -r response
        if [[ "$response" =~ ^[Yy]$ ]]; then
            if [ -f "${SCRIPT_DIR}/mnt.zip" ]; then
                log "INFO" "mnt.zip 압축 해제 중..."
                unzip -o "${SCRIPT_DIR}/mnt.zip" -d "${SCRIPT_DIR}/"
                if [ -d "$SOURCE_DIR" ]; then
                    log "INFO" "압축 해제 완료!"
                else
                    log "ERROR" "압축 해제 후에도 소스 디렉토리를 찾을 수 없습니다."
                    exit 1
                fi
            else
                log "ERROR" "mnt.zip 파일을 찾을 수 없습니다: ${SCRIPT_DIR}/mnt.zip"
                exit 1
            fi
        else
            exit 1
        fi
    fi
    
    # 필수 명령어 확인
    for cmd in rsync mkdir date; do
        if ! command -v "$cmd" >/dev/null 2>&1; then
            log "ERROR" "필수 명령어가 없습니다: $cmd"
            exit 1
        fi
    done
    
    log "DEBUG" "환경 검증 완료"
}

create_backup() {
    local target_dir="$1"
    local backup_dir="${target_dir}/.backup/$(date +%Y%m%d_%H%M%S)"
    
    if [ -d "$target_dir" ] && [ "$(ls -A "$target_dir" 2>/dev/null | wc -l)" -gt 0 ]; then
        log "INFO" "기존 파일 백업 중: $backup_dir"
        
        if [ "$DRY_RUN" = false ]; then
            mkdir -p "$backup_dir"
            cp -r "$target_dir"/* "$backup_dir/" 2>/dev/null || true
            log "INFO" "백업 완료: $backup_dir"
        else
            log "INFO" "[DRY-RUN] 백업 예정: $backup_dir"
        fi
    fi
}

sync_skills() {
    local source="$1"
    local target="$2"
    
    log "INFO" "스킬 동기화 시작"
    log "INFO" "소스: $source"
    log "INFO" "대상: $target"
    
    if [ "$DRY_RUN" = true ]; then
        log "INFO" "=== DRY RUN 모드 ===\n"
    fi
    
    # 대상 디렉토리 생성
    if [ "$DRY_RUN" = false ]; then
        mkdir -p "$target"
    else
        log "INFO" "[DRY-RUN] 디렉토리 생성 예정: $target"
    fi
    
    # 각 언어별 스킬 동기화
    local synced_count=0
    local total_count=${#LANGUAGES[@]}
    
    for lang in "${LANGUAGES[@]}"; do
        local source_skill="${source}/user/${lang}"
        local target_skill="${target}/user/${lang}"
        
        if [ -d "$source_skill" ]; then
            log "DEBUG" "동기화 중: $lang"
            
            if [ "$DRY_RUN" = false ]; then
                mkdir -p "$(dirname "$target_skill")"
                rsync -a --delete "$source_skill/" "$target_skill/"
                synced_count=$((synced_count + 1))
                
                # 진행률 표시
                local progress=$((synced_count * 100 / total_count))
                printf "\r${BLUE}진행률: [%-20s] %d%% (%d/%d)${NC}" \
                    $(printf "=%.0s" $(seq 1 $((progress / 5)))) \
                    "$progress" "$synced_count" "$total_count"
            else
                log "INFO" "[DRY-RUN] 동기화 예정: $source_skill → $target_skill"
                synced_count=$((synced_count + 1))
            fi
        else
            log "WARN" "스킬을 찾을 수 없습니다: $source_skill"
        fi
    done
    
    if [ "$DRY_RUN" = false ]; then
        echo "" # 새 줄
    fi
    
    log "INFO" "동기화 완료: $synced_count/$total_count 스킬"
}

setup_logging() {
    local target_dir="$1"
    LOG_FILE="${target_dir}/sync.log"
    
    if [ "$DRY_RUN" = false ]; then
        mkdir -p "$(dirname "$LOG_FILE")"
        log "INFO" "로그 파일: $LOG_FILE"
    fi
}

show_summary() {
    local target_dir="$1"
    
    echo ""
    log "INFO" "🎉 동기화 완료!"
    echo ""
    
    if [ "$DRY_RUN" = false ]; then
        echo -e "${GREEN}✅ 설치 위치:${NC} $target_dir"
        echo -e "${GREEN}✅ 로그 파일:${NC} $LOG_FILE"
        echo ""
        echo -e "${CYAN}📚 사용법:${NC}"
        echo "  Claude Code에서 다음과 같이 요청하세요:"
        echo '  • "React로 할일 관리 앱을 만들어줘"'
        echo '  • "PHP Laravel로 API를 만들어줘"'
        echo '  • "JavaScript 에러 처리를 개선해줘"'
        echo ""
        echo -e "${YELLOW}💡 팁:${NC}"
        echo '  "내 스킬을 사용해서..."라고 명시적으로 요청하면 더 정확합니다.'
    else
        echo -e "${YELLOW}💡 실제 동기화를 위해 --dry-run 옵션 없이 실행하세요.${NC}"
    fi
}

# ========================================
# 📋 메인 실행 로직
# ========================================

main() {
    local target_dir="${CLAUDE_SKILLS_ROOT:-$DEFAULT_TARGET_DIR}"
    
    # 명령행 인수 파싱
    while [[ $# -gt 0 ]]; do
        case $1 in
            --target)
                target_dir="$2"
                shift 2
                ;;
            --dry-run)
                DRY_RUN=true
                shift
                ;;
            --verbose)
                VERBOSE=true
                shift
                ;;
            -h|--help)
                show_help
                exit 0
                ;;
            *)
                log "ERROR" "알 수 없는 옵션: $1"
                echo "사용법: $0 --help"
                exit 1
                ;;
        esac
    done
    
    # 메인 실행
    print_header
    
    log "INFO" "Claude Skills 동기화를 시작합니다..."
    
    validate_environment
    setup_logging "$target_dir"
    
    # 권한 확인
    if [ "$DRY_RUN" = false ] && [ ! -w "$(dirname "$target_dir")" ]; then
        log "ERROR" "대상 경로에 쓰기 권한이 없습니다: $(dirname "$target_dir")"
        log "INFO" "다른 경로를 지정하세요: $0 --target ~/my-claude-skills"
        exit 1
    fi
    
    create_backup "$target_dir"
    sync_skills "$SOURCE_DIR" "$target_dir"
    show_summary "$target_dir"
}

# 스크립트 실행
if [[ "${BASH_SOURCE[0]}" == "${0}" ]]; then
    main "$@"
fi