# 🎯 Claude Skills 사용 예시 가이드

## 📚 목차
- [기본 사용법](#기본-사용법)
- [실전 시나리오](#실전-시나리오)
- [팀 협업](#팀-협업)
- [고급 활용](#고급-활용)

## 🚀 기본 사용법

### 1. 새 프로젝트 시작하기

#### React 프로젝트 생성
```
사용자: "React로 할일 관리 앱을 만들어줘. TypeScript 사용하고 싶어."

Claude: [React 스킬 적용]
- 프로젝트 구조 생성 (src/components/, src/hooks/, src/utils/)
- TypeScript 설정 (tsconfig.json)
- 컴포넌트 패턴 적용 (useState, useEffect, custom hooks)
- 에러 핸들링 패턴 포함
- Jest + Testing Library 테스트 설정
```

#### PHP Laravel API 개발
```
사용자: "사용자 관리 API를 Laravel로 만들어줘."

Claude: [PHP 스킬 적용]
- Laravel 프로젝트 구조 생성
- User 모델과 마이그레이션 
- UserService 클래스 (비즈니스 로직)
- API 컨트롤러 (CRUD 엔드포인트)
- Result 패턴으로 에러 핸들링
- PHPUnit 테스트 코드
- API 문서화
```

### 2. 코드 리뷰 및 개선

#### 에러 핸들링 개선
```
사용자: "이 JavaScript 코드의 에러 처리를 개선해줘."

// 기존 코드
async function fetchUserData(id) {
    const response = await fetch(`/api/users/${id}`);
    return response.json();
}

Claude: [JavaScript 스킬의 에러 핸들링 패턴 적용]
- try-catch with custom errors
- Retry 로직 구현
- 타입 안전성 검증
- 로깅 및 모니터링 추가
```

#### 테스트 커버리지 향상
```
사용자: "이 함수에 대한 포괄적인 테스트를 작성해줘."

Claude: [해당 언어 스킬의 테스팅 패턴 적용]
- 단위 테스트 (정상/실패 케이스)
- 통합 테스트
- Mock 활용
- Data Provider 패턴
- Edge case 커버리지
```

## 💼 실전 시나리오

### 시나리오 1: 전자상거래 사이트 개발

#### 프론트엔드 (React)
```
사용자: "상품 목록 페이지를 만들어줘. 검색, 필터링, 페이지네이션 포함해서."

Claude 응답:
1. React 스킬 템플릿의 프로젝트 구조 적용
2. 커스텀 훅 생성 (useProductSearch, useFilters, usePagination)
3. 성능 최적화 (useMemo, useCallback, React.memo)
4. 에러 경계 컴포넌트
5. 접근성 고려 (ARIA 라벨, 키보드 네비게이션)
6. 반응형 디자인
```

#### 백엔드 (PHP Laravel)
```
사용자: "상품 관리 API를 만들어줘. 이미지 업로드도 포함해서."

Claude 응답:
1. PHP 스킬 템플릿의 서비스 패턴 적용
2. Product 모델과 관계 정의
3. ProductService (비즈니스 로직 분리)
4. 이미지 처리 (validation, resize, optimization)
5. API 리소스 (JSON 응답 형식화)
6. 인증/인가 미들웨어
7. 캐싱 전략
```

### 시나리오 2: 실시간 채팅 앱

#### WebSocket 구현 (Node.js)
```
사용자: "실시간 채팅 서버를 만들어줘."

Claude 응답:
1. Node.js 스킬 템플릿의 Express 구조 적용
2. Socket.IO 설정
3. 채팅방 관리 클래스
4. 메시지 검증 및 sanitization
5. 연결 상태 관리
6. 에러 핸들링 (연결 끊김, 재연결)
7. 부하 테스트 코드
```

#### 프론트엔드 (Vue.js)
```
사용자: "채팅 인터페이스를 Vue로 만들어줘."

Claude 응답:
1. Vue 스킬 템플릿의 Composition API 패턴 적용
2. WebSocket 연결 관리 composable
3. 메시지 상태 관리 (Pinia)
4. 실시간 UI 업데이트
5. 타이핑 인디케이터
6. 이모지 지원
7. 오프라인 처리
```

### 시나리오 3: 데이터 분석 대시보드

#### 백엔드 (Python Django)
```
사용자: "매출 데이터 분석 API를 만들어줘."

Claude 응답:
1. Python 스킬 템플릿의 Django 구조 적용
2. 데이터 모델 설계 (시계열 데이터)
3. 분석 서비스 클래스 (pandas, numpy 활용)
4. 캐싱 전략 (Redis)
5. 비동기 처리 (Celery)
6. API 문서화 (DRF)
7. 성능 최적화
```

## 👥 팀 협업

### 코드 리뷰 자동화
```
사용자: "이 PR의 코드를 리뷰해줘. 우리 팀 컨벤션에 맞는지 확인해줘."

Claude 응답:
1. 해당 언어 스킬의 팀 컨벤션 체크리스트 적용
2. 코드 스타일 검증 (ESLint, PSR, PEP8)
3. 보안 취약점 확인
4. 성능 최적화 제안
5. 테스트 커버리지 확인
6. 문서화 상태 점검
```

### 온보딩 가이드 생성
```
사용자: "신입 개발자를 위한 프로젝트 구조 설명서를 만들어줘."

Claude 응답:
1. 각 스킬 템플릿의 프로젝트 구조 예시 활용
2. 폴더별 역할 설명
3. 주요 파일들의 목적
4. 개발 워크플로우
5. 테스팅 전략
6. 배포 과정
```

## 🎓 고급 활용

### 마이크로서비스 아키텍처
```
사용자: "마이크로서비스로 전자상거래 시스템을 설계해줘."

Claude 응답:
1. 각 서비스별로 적절한 언어 스킬 적용
   - 사용자 서비스: Java/Spring Boot
   - 상품 서비스: Node.js/Express
   - 주문 서비스: Python/Django
   - 결제 서비스: Go

2. 공통 패턴 적용
   - API Gateway 설정
   - 서비스 간 통신 (gRPC/REST)
   - 분산 트랜잭션 관리
   - 모니터링 및 로깅
   - CI/CD 파이프라인
```

### 성능 최적화 컨설팅
```
사용자: "이 웹사이트가 너무 느려. 성능을 개선해줘."

Claude 응답:
1. 프론트엔드 최적화 (HTML/CSS/JS 스킬 적용)
   - Critical CSS 인라인
   - 이미지 최적화
   - 코드 분할
   - 캐싱 전략

2. 백엔드 최적화 (해당 언어 스킬 적용)
   - 데이터베이스 쿼리 최적화
   - 캐싱 레이어 추가
   - API 응답 최적화
   - 메모리 관리

3. 인프라 최적화 (Docker/K8s 스킬 적용)
   - 컨테이너 최적화
   - 로드 밸런싱
   - CDN 설정
```

## 📝 실전 커맨드 예시

### 개발 환경 설정
```bash
# 1. 스킬 동기화
./setup_claude_skills.sh

# 2. 프로젝트 시작
Claude: "React + TypeScript 프로젝트를 시작해줘"
# → React 스킬의 프로젝트 구조와 설정이 자동 적용됨

# 3. 개발 중 문제 해결
Claude: "API 호출에서 429 에러가 나고 있어. 어떻게 처리하지?"
# → JavaScript 스킬의 retry 로직과 에러 핸들링 패턴 적용됨
```

### 코드 품질 관리
```bash
# 코드 리뷰 요청
Claude: "이 컴포넌트를 리뷰해줘. 성능과 접근성 측면에서"
# → React 스킬의 체크리스트와 베스트 프랙티스 적용됨

# 테스트 작성 요청  
Claude: "이 서비스 클래스에 대한 테스트를 작성해줘"
# → PHP 스킬의 PHPUnit 패턴과 데이터 프로바이더 적용됨
```

## 🎯 효과적인 활용 팁

### 1. 구체적인 요청하기
```
❌ "웹사이트 만들어줘"
✅ "React로 할일 관리 앱을 만들어줘. TypeScript 사용하고, 상태관리는 Context API로 해줘"
```

### 2. 컨텍스트 제공하기
```
❌ "에러 처리해줘"
✅ "API 호출 중 네트워크 에러와 401/403 인증 에러를 구분해서 처리해줘"
```

### 3. 점진적 개선 요청
```
✅ "먼저 기본 구조 만들어줘"
✅ "이제 에러 핸들링 추가해줘" 
✅ "테스트 코드도 작성해줘"
✅ "성능 최적화도 해줘"
```

이제 Claude Skills가 제공하는 풍부한 템플릿과 패턴을 활용해서 더 효율적이고 품질 높은 개발을 해보세요! 🚀